/* $Id: adjlist.c,v 1.14 2007/01/25 10:02:10 mcv21 Exp $*/
/*
 * This file is part of the library of graph analysis and disease
 * simulation functions submitted along with the thesis "Spacial Spread
 * of Farm Animal Diseases" for the degree of Doctor of Philosophy at the
 * University of Cambridge. 
 *
 * The library is Copyright (C) 2007 Matthew Vernon <matthew@debian.org>
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (as gpl.txt); if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include <stdlib.h>
#include <stdio.h>

#include "ferror.h"
#include "adjlist.h"

struct adjl *adjl_new(const int n)
{
  struct adjl *l;

  if(NULL==(l=malloc(sizeof(*l))))
    fatal_error("Unable to allocate list structure",NULL,1);
  l->n=0;
  l->id=-1;

  if(n>0){ /*calloc makes everything zero*/
    if(NULL==(l->neighb=calloc(n,sizeof(*(l->neighb)))))
      fatal_error("Unable to allocate list",NULL,1);
    l->length=n;
  } else {
    l->length=0;
    l->neighb=NULL;
  }

  return(l);
}

/*return 0 if edge already existed, 1 otherwise*/
int adjl_add(struct adjl *l, int new)
{
  /*don't add duplicated neighbours*/
  if(adjl_neighbour(l,new)) return(0);

  /*If we're about to overflow, allocate double the initial space
   *this is a crude optimisation to avoid continually calling realloc
   */
  if(1 + l->n > l->length){
    /* 0*2 would still be zero...*/
    if(l->length==0) l->length=1;
    else l->length*=2;
    /*ancient systems may crash if l->neighb is NULL, but ANSI allows
     *this
     */
    if(NULL==(l->neighb=realloc(l->neighb,l->length*sizeof(*(l->neighb)))))
	fatal_error("Unable to extend adjacency list",NULL,1);
  }
  /*arrays are 0-indexed, remember*/
  l->neighb[l->n]=new;
  l->n++;
  return(1);
}

int adjl_neighbour(const struct adjl *haystack, const int needle)
{
  int a;

  for(a=0;a<haystack->n;a++)
    if(needle==haystack->neighb[a]) return(1);

  return(0);
}

void adjl_remove(struct adjl *haystack,const int needle)
{
  int a,found=0;

  for(a=0;a<haystack->n;a++)
    if(found)
      haystack->neighb[a-1]=haystack->neighb[a];
    else if(needle==haystack->neighb[a]) found=1;
  haystack->n--; /*remove the extra space*/
  
  if(0==found) fatal_error("Needle not found to remove",NULL,0);
}


struct adjl *adjl_symmetrise(const struct adjl *g, const int n)
{
  struct adjl *l;
  int i,j,tmp;

  l=xcalloc(n,sizeof(*l));

  for(i=0;i<n;i++) l[i].id=i;

  /*bidirectionalise every edge*/
  for(i=0;i<n;i++){
    for(j=0;j<g[i].n;j++){
      tmp=g[i].neighb[j];
      adjl_add(&l[i],tmp);
      adjl_add(&l[tmp],i);
    }
  }

  return(l);
}

struct adjl *adjl_reverse(const struct adjl *g, const int n)
{
  struct adjl *l;
  int i, j;

  l=xcalloc(n,sizeof(*l));

  for(i=0;i<n;i++) l[i].id=i; 
  /*We want y->x in l for every x->y in g*/
  for(i=0;i<n;i++)
    for(j=0;j<g[i].n;j++)
      adjl_add(&l[g[i].neighb[j]],i);

  return(l);
}
