/* $Id: fheap.c,v 1.10 2007/01/25 10:02:10 mcv21 Exp $ */
/*
 * This file is part of the library of graph analysis and disease
 * simulation functions submitted along with the thesis "Spacial Spread
 * of Farm Animal Diseases" for the degree of Doctor of Philosophy at the
 * University of Cambridge. 
 *
 * The library is Copyright (C) 2007 Matthew Vernon <matthew@debian.org>
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (as gpl.txt); if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include <stdio.h>
#include <stdlib.h>

#include "gads.h"

static void fh_link(struct fh_node *c, struct fh_node *p);
static void fh_reinsert(struct fh_node *node, struct fh_heap *heap);
static void fh_consolidate(struct fh_heap *h);
static void fh_cut(struct fh_node *c, struct fh_node *p, struct fh_heap *h);
static void fh_cascade(struct fh_node *node, struct fh_heap *h);


struct fh_node *fh_new_node(void)
{
  struct fh_node *new;

  if(NULL==(new=(struct fh_node *)(malloc(sizeof(struct fh_node))))){
    fprintf(stderr,"Error: out of memory in fh_new_node.\n");
    exit(1);
  }

  new->right=new;
  new->left=new;
  new->mark=0;
  new->key=-1;
  new->degree=0;
  new->id=0;
  new->parent=NULL;
  new->child=NULL;
  return(new);
}

struct fh_heap *fh_new_heap(void)
{
  struct fh_heap *new;

  if(NULL==(new=(struct fh_heap *)(malloc(sizeof(struct fh_heap))))){
    fprintf(stderr,"Error: out of memory in fh_new_heap.\n");
    exit(1);
  }

  new->min=NULL;
  new->n=0; 

  return(new);
}

/*insert node to the left of heap->min in the root list
 *and adjust heap->min if needed
 */
void fh_insert_node(struct fh_node *new, struct fh_heap *heap)
{
  if((NULL==new)||(NULL==heap)){
    fprintf(stderr,"Error: null value passed to fh_insert_node.\n");
    exit(1);
  }

  if(heap->min){
    heap->min->left->right=new;
    new->left=heap->min->left;
    new->right=heap->min;
    heap->min->left=new;
  }
  if((NULL==heap->min)||(new->key<heap->min->key))
    heap->min=new;
  heap->n++;
}

/*make c a child of p*/
static void fh_link(struct fh_node *c, struct fh_node *p)
{
  /*remove c from the root list of H
   *The checks are needed because we linearised the root list
   */
  if(c->left!=NULL) c->left->right=c->right;
  if(c->right!=NULL)c->right->left=c->left;

  if(p->child){
    p->child->right->left=c;
    c->right=p->child->right;
    p->child->right=c;
    c->left=p->child;
  }
  else{
    c->left=c;
    c->right=c;
    p->child=c;
  }
  p->degree++;
  c->mark=0;
  c->parent=p;
}

/*re-inserts n into the root list of h*/
static void fh_reinsert(struct fh_node *node, struct fh_heap *heap)
{
  node->mark=0; /*clear any mark*/
  node->parent=NULL;
  node->right=node;
  node->left=node;
  heap->n--; /*since insert will increment this*/
  fh_insert_node(node,heap);
}

static void fh_consolidate(struct fh_heap *h)
{
  /*allowed in C99; otherwise use malloc()*/
  struct fh_node *dega[h->n]; /*Array of node degrees*/
  struct fh_node *x, *y, *w, *tmp;
  int d, i;

  /*if we have no nodes, don't do anything*/
  if(NULL==h->min) return;

  for(i=0;i<h->n;i++) dega[i]=NULL;

  /*break the circle of root nodes, to simplify finding our
   *end-point (our starting node may not be a root node by the time
   *we're done
   */
  h->min->left->right=NULL;
  h->min->left=NULL;
  
  w=h->min;
  do{ /*while w is not NULL*/
    x=w;
    d=x->degree;
    w=w->right;
    while(dega[d]!=NULL){
      y=dega[d];
      if(x->key>y->key){ /*swap y and x around*/
	tmp=x; x=y; y=tmp;
      }
      if(y==w) w=w->right; /*move along*/
      fh_link(y,x); /*make y a child of x*/
      dega[d]=NULL; /*since x has degree x+1 now*/
      d++;
    } /*so now dega[a]==NULL, we can make x equal to dega[a]*/
    dega[d]=x;
  } while(w!=NULL);
  
  /*now rebuild the root node list*/
  h->min=NULL;
  for(i=0;i<h->n;i++) if(dega[i]) fh_reinsert(dega[i],h);
}

/*return the key from the extracted node*/
struct fh_node *fh_extract_min(struct fh_heap *heap)
{
  struct fh_node *ans; /* The answer :) */

  if(NULL==heap->min) return(NULL);
  
  ans=heap->min; 
  /*add the child node(s) of ans to the root list
   *note that their parent field will be incorrect,
   *but we don't care, since consolidate will fix that 
   *in a bit
   */
  if(heap->min->child){
    heap->min->left->right=heap->min->child->right;
    heap->min->child->right->left=heap->min->left;
    heap->min->left=heap->min->child;
    heap->min->child->right=heap->min;
  }
  /*now remove ans from the root list*/
  ans->right->left=ans->left;
  ans->left->right=ans->right;
  if(ans->right==ans) /*i.e. answer was the only node left*/
    heap->min=NULL;
  else heap->min=ans->right;
  ans->left=ans->right=ans;
  heap->n--;
  fh_consolidate(heap);
  return(ans);
}

static void fh_cut(struct fh_node *c, struct fh_node *p, struct fh_heap *h)
{
  /*if the child is p->child also, make p->child point
   *elsewhere
   */
  if(p->child==c) p->child=c->right;
  /*excise c*/
  c->right->left=c->left;
  c->left->right=c->right;
  /*if p->child is still c, then it was the only child of p*/
  if(p->child==c) p->child=NULL;
  p->degree--;
  /*add c to root list of h*/
  fh_reinsert(c,h);
}

/*cut nodes in the parent list, until we reach the root list
 *a node is marked the first time it loses a child, and will be
 *cut the next time it loses a child subtree, and moved to the 
 *root list, where it will be re-balenced during the next
 *consolidation
 */
static void fh_cascade(struct fh_node *node, struct fh_heap *h)
{
  struct fh_node *p;

  p=node->parent;

  if(p){
    if(0==node->mark) node->mark=1;
    else{
      fh_cut(node,p,h);
      fh_cascade(p,h);
    }
  }
}

void fh_decrease_key(struct fh_node *node, struct fh_heap *heap, 
		     const int val)
{
  struct fh_node *par;

  if(val>node->key){
    fprintf(stderr,"Error: value larger than node key.\n");
    exit(1);
  }
  node->key=val;
  par=node->parent;
  if((par)&&(par->key > node->key)){
    fh_cut(node,par,heap);
    fh_cascade(par,heap);
  }
  if(node->key < heap->min->key) heap->min=node;
}
