/*$Id: gens.c,v 1.10 2007/01/25 10:02:10 mcv21 Exp $*/
/*
 * This file is part of the library of graph analysis and disease
 * simulation functions submitted along with the thesis "Spacial Spread
 * of Farm Animal Diseases" for the degree of Doctor of Philosophy at the
 * University of Cambridge. 
 *
 * The library is Copyright (C) 2007 Matthew Vernon <matthew@debian.org>
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (as gpl.txt); if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include <string.h>
/*stdint.h is a C99ism, which solaris9 lacks*/
#ifdef SOLARIS
#include <limits.h>
#else
#include <stdint.h>
#endif

#include "ferror.h"

#include "gens.h"
#include "g_bm.h"
#include "g_im.h"
#include "g_al.h"
#include "g_special.h"

struct gennet *bitmatrix_create(const int n)
{
  int n2,a;
  n2=BM_LEN(n);
  struct bitmatrix *b;

  if ( n*n2 > SIZE_MAX / sizeof **b->net)
    fatal_error("requested matrix is too large",NULL,0);
 
  b=xmalloc(sizeof(*b));
  b->general.vtable=&vtable_for_bitmatrix;
  b->net=xmalloc(n*sizeof(*b->net));
  b->net=memset(b->net,0,n*sizeof(*b->net));
  b->net[0]=xmalloc((size_t)n*n2*sizeof(**b->net));
  b->net[0]=memset(b->net[0],0,n*n2*sizeof(**b->net));
  for (a=1;a<n;a++) b->net[a]=(b->net[0])+(a*n2);
  b->n=n;
  return(&b->general);
}

struct gennet *intmatrix_create(const int n)
{
  struct intmatrix *m;
  int a;

  if ( n*n > SIZE_MAX / sizeof **m->net)
    fatal_error("requested matrix is too large",NULL,0);
  m=xmalloc(sizeof(*m));
  m->general.vtable=&vtable_for_intmatrix;
  m->net=xmalloc(n*sizeof(*m->net));
  m->net=memset(m->net,0,n*sizeof(*m->net));
  m->net[0]=xmalloc((size_t)n*n*sizeof(**m->net));
  m->net[0]=memset(m->net[0],0,n*n*sizeof(**m->net));
  for(a=1;a<n;a++) m->net[a]=(m->net[0])+(a*n);
  m->n=n;
  return(&m->general);
}

struct gennet *adjlist_create(const int n)
{
  struct adjlnet *m;
  int a;

  if ( n > SIZE_MAX / sizeof(*m->net))
    fatal_error("requested list of adjacency lists is too large",NULL,0);
  m=xmalloc(sizeof(*m));
  m->general.vtable=&vtable_for_adjlistnet;

  m->net=xcalloc(n,sizeof(*m->net));
  for(a=0;a<n;a++) m->net[a].id=a;
  m->sorted=1; /*empty network doesn't need sorting*/
  return(&m->general);
}

struct gennet *fullgraph_create(const int n)
{
  struct fullgraph *f;

  f=xmalloc(sizeof(*f));
  f->general.vtable=&vtable_for_fullgraph;
  f->n=n;

  return(&f->general);
}

struct gennet *nullgraph_create(const int n)
{
  struct gennet *g;

  g=xmalloc(sizeof(*g));
  g->vtable=&vtable_for_nullgraph;
  return(g);
}

struct gennet *type_create(const int n, const net_desired_t type)
{
  switch(type){
  case G_BITM:
    return bitmatrix_create(n);
  case G_INTM:
    return intmatrix_create(n);
  case G_ADJL:
    return adjlist_create(n);
  case G_FULL:
    return fullgraph_create(n);
  case G_NULL:
    return nullgraph_create(n);
  default:
    fatal_error("Must specify a network type",NULL,0);
  }
}
