/*$Id: specalg.c,v 1.11 2007/01/25 10:02:10 mcv21 Exp $*/
/*
 * This file is part of the library of graph analysis and disease
 * simulation functions submitted along with the thesis "Spacial Spread
 * of Farm Animal Diseases" for the degree of Doctor of Philosophy at the
 * University of Cambridge. 
 *
 * The library is Copyright (C) 2007 Matthew Vernon <matthew@debian.org>
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (as gpl.txt); if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

/*This file contains graph algorithms that do need to peer underneath the
 *bonnet, for whatever reason (typically, a substantial efficiency gain)
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include "ferror.h"
#include "gens.h"
#include "gads.h"
#include "census.h"
#include "utils.h"
#include "gsalgs.h"

#include "g_al.h"
#include "g_bm.h"
#include "g_im.h"

static int adjl_triad_code(const struct adjl *l,const int v, 
			   const int u, const int w);

typedef enum {
  S,
  I,
  R } state_t;

void sir_net(struct gennet *g, const int n, const int istart, 
	     const double risk, const int remain, const int t,
	     FILE *out)
{
  struct adjlnet *l=g->vtable->type==G_ADJL?(struct adjlnet *)g:NULL;
  state_t states[n];
  int time[n]; /*how long this node has left in this state*/
  struct dll *inodes;
  struct dlln *j, *oldj;
  struct sll *new;
  int a,b,c,x;
  double xtmp;
  int s=n-istart,i=istart,r=0;

  if((1<risk)||(0>risk)){
    fprintf(stderr,"risk value %f not reasonable\n",risk);
    exit(1);
  }
  if(0>=remain){
    fprintf(stderr,"infectious period %d makes no sense\n",remain);
    exit(1);
  }

  inodes=new_dll();
  new=new_sll();

  for(a=0;a<n;a++) states[a]=S;
  /*initial infection*/
  for(a=0;a<istart;a++)
    for(;;){ /*XXX want to use round() really*/
      xtmp=drand48()*n;
      x=floor(xtmp);
      if(S==states[x]){
	states[x]=I;
	time[x]=remain;
	dll_append(inodes,x);
	break;
      }
    }


  for(a=0;a<t;a++){
    /*Beginning of an iteration, so output current figures*/
    fprintf(out,"%d %d %d %d\n",a,s,i,r);
    if(0==i){
      /*clear up*/
      free(inodes);
      free(new);
      return;
    }
    /*Iterate over all infected nodes, seeing which neighbours
     *they infect
     */
    j=inodes->top;
    if(NULL==j) break;
    do{
      b=j->x;
      if(0==time[b]){
	/*This node has now recovered*/
	states[b]=R; r++; i--;
	oldj=j;
	j=oldj->next;
	dll_remove(inodes,oldj);
      }else{
	if(l){
	  for(x=0;x<l->net[b].n;x++){
	    c=l->net[b].neighb[x];
	    if(S==states[c]){
	      if(drand48()<=risk){
		/*infected!*/
		sll_append(new,c);
		states[c]=I; i++; s--;
		time[c]=remain;
	      }
	    }
	  }
	}else{
	  for(c=g->vtable->next_neighb(g,b,-1);c!=-1;
	      c=g->vtable->next_neighb(g,b,c)){
	    if(S==states[c]){
	      if(drand48()<=risk){
		/*infected!*/
		sll_append(new,c);
		states[c]=I; i++; s--;
		time[c]=remain;
	      }
	    }
	  }
	} 
	time[b]--;
	j=j->next;
      }
    }
    while((j!=inodes->bottom)&&(j!=NULL));
    /*Deal with newly-infected nodes
     *remove them from new and put them into inodes
     */
    for(;;){
      x=sll_remove(new);
      if(-1==x) break;
      dll_append(inodes,x);
    }
  }
  /*clear up*/
  free(inodes);
  free(new);
}

/*This implements the algorithm in Batagelj:2001
 *And should run in O(m) time, where m is the number of edges
 *it requires an adjacency-list-style network structure
 *so the first step is to convert the passed-in network 
 *if necessary
 */
void triad_census(struct gennet *g, const int n, triad_count tc)
{
  struct adjlnet *an=g->vtable->type==G_ADJL?(struct adjlnet *)g:NULL;
  int i,u,v,w;
  int *s, ns;
  int needtofree=0;
  struct adjl *sym;
  uint64_t tmp,n64;

  if(NULL==an){
    an=(struct adjlnet *)al_convert(g,n);
    needtofree=1;
  }
  an->general.vtable->sort(&an->general);

  sym=adjl_symmetrise(an->net,n);
  for(v=0;v<n;v++)
    qsort(sym[v].neighb,sym[v].n,sizeof(int),int_cmp);

  for(i=0;i<TTYPES;i++) tc[i]=0;

  for(v=0;v<n;v++){
    for(u=0;u<sym[v].n;u++)
      if(v < sym[v].neighb[u]){
	s=int_union(sym[v].neighb,sym[v].n,
		    sym[sym[v].neighb[u]].neighb,
		    sym[sym[v].neighb[u]].n,&ns);
	if(adjl_neighbour(&an->net[v],sym[v].neighb[u])&&
	   adjl_neighbour(&an->net[sym[v].neighb[u]],v))
	  tc[T102]+=(n-ns);
	else tc[T012]+=(n-ns);
	for(w=0;w<ns;w++){
	  /*line 2.1.1 of the paper excludes u,v from S; we discount them here*/
	  if( (s[w]!=v)&&(s[w]!=sym[v].neighb[u]) &&
	      /*only count the triad if it's in canonical order*/
	      ( (sym[v].neighb[u] < s[w]) ||
		( (v < s[w]) && (s[w] < sym[v].neighb[u])
		  && (0==adjl_neighbour(&sym[v],s[w])) ) ) ) {
	    tc[adjl_triad_code(an->net,
			       v,sym[v].neighb[u],s[w])]++;
	  }
	}
	free(s);
      }
  }
  /*calculate the T003 [==tc[0]] by elimination*/
  n64=n;
  tmp=(n64*(n64-1)*(n64-2))/6;
  for(i=1;i<TTYPES;i++) tmp-=tc[i];
  tc[T003]=tmp;
  adjlstar_free(sym,n);
  if(needtofree) an->general.vtable->free(&an->general);
}

static int adjl_triad_code(const struct adjl *l,const int v, 
			   const int u, const int w)
{
  int t=0;
  if(adjl_neighbour(&l[v],u)) t|=AB;
  if(adjl_neighbour(&l[u],v)) t|=BA;
  if(adjl_neighbour(&l[v],w)) t|=AC;
  if(adjl_neighbour(&l[w],v)) t|=CA;
  if(adjl_neighbour(&l[u],w)) t|=BC;
  if(adjl_neighbour(&l[w],u)) t|=CB;

  return triad_table[t];
}

/*This also needs an adjacency-style network structure
 *this approach is quicker than simply iterating over every
 *pair of nodes. instead for each node >0, we consider the
 *in- and out- node lists of nodes <that node:
 *M==|in&&out| and A==|inXORout|
 */
void dyad_census(struct gennet *g, const int n, dyad_count *dc)
{
  struct adjl *in;
  struct adjlnet *out=g->vtable->type==G_ADJL?(struct adjlnet *)g:NULL;
  int i,mtmp,atmp,*dump;
  int needtofree=0;
  int64_t n64;

  if(NULL==out){
    out=(struct adjlnet *)al_convert(g,n);
    needtofree=1;
  }
  out->general.vtable->sort(&out->general);

  /*generate an in-list adjacency list, and zero the dyad counters*/
  in=adjl_reverse(out->net,n);
  for(i=0;i<n;i++)
    qsort(in[i].neighb,in[i].n,sizeof(int),int_cmp);
  dc->m=0;
  dc->a=0;

  /*only interested in j<i, so start with i=1
   *we only care about the length of the result, so
   *free() it afterewards
   */
  for(i=1;i<n;i++){
    dump=int_and(out->net[i].neighb,out->net[i].n,
		 in[i].neighb,in[i].n,i,&mtmp);
    free(dump); 
    dc->m+=mtmp;
    dump=int_xor(out->net[i].neighb,out->net[i].n,
		 in[i].neighb,in[i].n,i,&atmp);
    free(dump);
    dc->a+=atmp;
  }
  /*calculate the null count by elimination*/
  n64=n;
  /*n(n-1)/2 dyads in a digraph*/
  dc->n=(n64*(n64-1))/2;
  dc->n-=(dc->m + dc->a);
  adjlstar_free(in,n); /*avoid leaking*/
  if(needtofree)out->general.vtable->free(&out->general);
}


/*rewire the network to match the dyad census dc,
 *whilst preserving the 2-dimensional degree distribution
 *uses adjacency-lists representation
 *NB! if you pass in another type of network, then the adjacency-list
 *form is returned, but the net you passed in IS NOT freed. If you
 *don't intend using that again, free it yourself!
 */
struct gennet *dyad_rewire(struct gennet *g,const int n, 
			   const dyad_count *dc)
{
  struct adjlnet *l=g->vtable->type==G_ADJL?(struct adjlnet *)g:NULL;
  dyad_count tmpdc;
  int a,b,c,d,i,fail=0,mf,atod;

  if(NULL==l)
    l=(struct adjlnet *)al_convert(g,n);

  dyad_census(&l->general,n,&tmpdc);
  mf=abs(tmpdc.m - dc->m); /*heuristic to prevent infinite looping*/
  if(n>mf)mf=n; /*quite a generous heuristic ;) */

  while(tmpdc.m<dc->m){
    if (fail>mf)
      fatal_error("Too much looping",NULL,0);
    a=floor((drand48()*(n-1))+0.5); /*XXX round() would be better*/
    if(0==l->net[a].n){fail++; continue;}
    i=floor((drand48()*((l->net[a].n)-1))+0.5);
    b=l->net[a].neighb[i];
    if((adjl_neighbour(&l->net[b],a))||(a==b)){
      fail++; continue;}
    if(0==l->net[b].n){fail++; continue;}
    i=floor((drand48()*((l->net[b].n)-1))+0.5);
    c=l->net[b].neighb[i];
    if((adjl_neighbour(&l->net[c],b))||(a==c)||(b==c)){
      fail++; continue;}
    if(0==l->net[c].n){fail++; continue;}
    i=floor((drand48()*((l->net[c].n)-1))+0.5);
    d=l->net[c].neighb[i];
    if(adjl_neighbour(&l->net[d],c)||(adjl_neighbour(&l->net[a],d))||
       (a==d)||(b==d)||(c==d)){
      fail++; continue;}
    atod=adjl_neighbour(&l->net[d],a);
    
    /*we have our 4 nodes. Now re-wire them*/
    adjl_remove(&l->net[a],b);
    adjl_remove(&l->net[c],d);
    adjl_add(&l->net[c],b);
    adjl_add(&l->net[a],d);
    /*We must ensure the adjacency lists stay sorted*/
    qsort(l->net[c].neighb,l->net[c].n,sizeof(int),int_cmp);
    qsort(l->net[a].neighb,l->net[a].n,sizeof(int),int_cmp);
    /*house-keeping - modify our dyad census, and note the success*/
    fail=0;
    if(atod){tmpdc.m+=2; tmpdc.a-=4; tmpdc.n+=2;}
    else {tmpdc.m+=1; tmpdc.a-=2; tmpdc.n+=1;}
  }
  return(&l->general);
}
